<?php
/**
 * Member View Page
 * GHAMECC Choir Management Platform
 */

// Include necessary files
require_once '../includes/auth.php';
require_once '../config/database.php';

// Ensure user is logged in
requireLogin();

// Check if member ID is provided
if (!isset($_GET['id']) || empty($_GET['id'])) {
    header("Location: list.php?error=invalid_id");
    exit;
}

$memberId = (int)$_GET['id'];

// Check if user has access to this member
if (!hasRole(['admin', 'manager']) && !hasMemberAccess($memberId)) {
    header("Location: list.php?error=permission_denied");
    exit;
}

// Get member details
$query = "SELECT m.*, s.name as society_name, c.name as circuit_name, d.name as diocese_name
          FROM members m
          LEFT JOIN societies s ON m.society_id = s.id
          LEFT JOIN circuits c ON s.circuit_id = c.id
          LEFT JOIN dioceses d ON c.diocese_id = d.id
          WHERE m.id = ?";

$member = fetchRows($query, [$memberId]);

// Check if member exists
if (empty($member)) {
    header("Location: list.php?error=member_not_found");
    exit;
}

$member = $member[0];

// Calculate age from date of birth if it exists
$age = null;
if (!empty($member['dob'])) {
    $dob = new DateTime($member['dob']);
    $now = new DateTime();
    $age = $now->diff($dob)->y;
}

// Check if member has a user account
$userAccount = fetchRows("SELECT u.* 
                         FROM users u 
                         WHERE u.member_id = ?", [$memberId]);

// Get attendance statistics
$attendanceStats = fetchRows("SELECT 
                             COUNT(DISTINCT event_id) as total_events,
                             SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present_count,
                             SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent_count,
                             SUM(CASE WHEN status = 'excused' THEN 1 ELSE 0 END) as excused_count,
                             ROUND((SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) / COUNT(DISTINCT event_id)) * 100, 1) as attendance_rate
                             FROM attendance 
                             WHERE member_id = ?", [$memberId]);

// Get recent attendance records
$recentAttendance = fetchRows("SELECT a.*
                              FROM attendance a
                              WHERE a.member_id = ?
                              ORDER BY a.id DESC
                              LIMIT 5", [$memberId]);

// Get financial records
$financialRecords = fetchRows("SELECT fr.*
                              FROM financial_records fr
                              WHERE fr.member_id = ?
                              ORDER BY fr.id DESC
                              LIMIT 10", [$memberId]);

// Calculate total contributions
$totalContributions = fetchRow("SELECT SUM(amount) as total
                              FROM financial_records
                              WHERE member_id = ?", [$memberId]);

// Set base path for includes
$basePath = '..';

// Add debug logging to help troubleshoot
require_once '../config/logger.php';
logInfo("Viewing member profile", [
    'member_id' => $memberId,
    'columns_available' => array_keys($member)
]);

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Member Profile - <?php echo htmlspecialchars($member['first_name'] . ' ' . $member['last_name']); ?> - GHAMECC</title>
    
    <?php include '../includes/favicon.php'; ?>
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        .profile-image {
            width: 150px;
            height: 150px;
            object-fit: cover;
            border-radius: 50%;
            border: 3px solid #fff;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        .profile-placeholder {
            width: 150px;
            height: 150px;
            border-radius: 50%;
            background-color: #f8f9fa;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 3rem;
            color: #6c757d;
            border: 3px solid #fff;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        .info-card {
            height: 100%;
            box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
        }
        .info-label {
            font-weight: 600;
            color: #4e73df;
        }
        .stat-card {
            padding: 0.5rem;
            border-radius: 0.35rem;
            background-color: #f8f9fc;
        }
        .stat-value {
            font-size: 1.5rem;
            font-weight: 700;
        }
        .stat-label {
            font-size: 0.8rem;
            color: #858796;
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <?php include '../includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <?php include '../includes/sidebar.php'; ?>
            
            <!-- Main Content -->
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Member Profile</h1>
                    <div>
                        <a href="list.php" class="btn btn-secondary me-2">
                            <i class="bi bi-arrow-left"></i> Back to Members List
                        </a>
                        <?php if (hasRole('admin') || hasRole('manager') || hasRole('secretary') || 
                                 hasRole('choirmaster') || hasRole('society_secretary') || hasRole('society_treasurer')): ?>
                            <a href="edit.php?id=<?php echo $memberId; ?>" class="btn btn-primary">
                                <i class="bi bi-pencil"></i> Edit Member
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Member Profile Header -->
                <div class="card shadow mb-4">
                    <div class="card-body">
                        <div class="row align-items-center">
                            <div class="col-md-2 text-center">
                                <?php if (!empty($member['profile_image'])): ?>
                                    <img src="../uploads/members/<?php echo htmlspecialchars($member['profile_image']); ?>" alt="Profile" class="profile-image">
                                <?php else: ?>
                                    <div class="profile-placeholder">
                                        <?php echo strtoupper(substr($member['first_name'], 0, 1) . substr($member['last_name'], 0, 1)); ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="col-md-10">
                                <h2><?php echo htmlspecialchars($member['first_name'] . ' ' . (!empty($member['middle_name']) ? $member['middle_name'] . ' ' : '') . $member['last_name']); ?></h2>
                                <p class="text-muted mb-1">
                                    <i class="bi bi-music-note"></i> <?php
                                    $voicePartName = !empty($member['part']) ? $member['part'] : 'N/A';
                                    if (strtolower($voicePartName) === 'other' && !empty($member['part_other'])) {
                                        $voicePartName = $member['part_other'];
                                    }
                                    echo htmlspecialchars($voicePartName);
                                    ?> |
                                    <i class="bi bi-building"></i> <?php echo htmlspecialchars($member['society_name'] ?? 'N/A'); ?> | 
                                    <i class="bi bi-geo"></i> <?php echo htmlspecialchars($member['circuit_name'] ?? 'N/A'); ?> Diocese
                                </p>
                                <p class="mb-1">
                                    <i class="bi bi-telephone"></i> <?php echo htmlspecialchars($member['contact_number'] ?? 'N/A'); ?>
                                </p>
                                <?php if (!empty($member['email'])): ?>
                                    <p class="mb-1">
                                        <i class="bi bi-envelope"></i> <?php echo htmlspecialchars($member['email']); ?>
                                    </p>
                                <?php endif; ?>
                                <p class="mb-0">
                                    <span class="badge bg-<?php echo ($member['membership_status'] ?? 'inactive') === 'active' ? 'success' : (($member['membership_status'] ?? 'inactive') === 'inactive' ? 'warning' : 'danger'); ?>">
                                        <?php echo ucfirst(htmlspecialchars($member['membership_status'] ?? 'Inactive')); ?>
                                    </span>
                                    <?php if (!empty($member['id'])): ?>
                                        <span class="badge bg-primary ms-2">ID: <?php echo htmlspecialchars($member['id']); ?></span>
                                    <?php endif; ?>
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <!-- Personal Information -->
                    <div class="col-md-6 mb-4">
                        <div class="card shadow info-card">
                            <div class="card-header py-3">
                                <h6 class="m-0 font-weight-bold text-primary">Personal Information</h6>
                            </div>
                            <div class="card-body">
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Full Name</div>
                                    <div class="col-md-8">
                                        <?php echo htmlspecialchars($member['first_name'] . ' ' . (!empty($member['middle_name']) ? $member['middle_name'] . ' ' : '') . $member['last_name']); ?>
                                    </div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Gender</div>
                                    <div class="col-md-8"><?php echo htmlspecialchars($member['sex'] ?? 'Not specified'); ?></div>
                                </div>
                                <?php if (!empty($member['dob'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Date of Birth</div>
                                    <div class="col-md-8">
                                        <?php echo date('F j, Y', strtotime($member['dob'])); ?> 
                                        <?php if (!is_null($age)): ?>
                                            (<?php echo $age; ?> years)
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($member['occupation'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Occupation</div>
                                    <div class="col-md-8"><?php echo htmlspecialchars($member['occupation']); ?></div>
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($member['address'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Address</div>
                                    <div class="col-md-8"><?php echo nl2br(htmlspecialchars($member['address'])); ?></div>
                                </div>
                                <?php endif; ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Contact</div>
                                    <div class="col-md-8">
                                        <div><?php echo htmlspecialchars($member['contact_number'] ?? 'N/A'); ?> (Primary)</div>
                                        <?php if (!empty($member['alt_contact_number'])): ?>
                                            <div><?php echo htmlspecialchars($member['alt_contact_number']); ?> (Alternative)</div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php if (!empty($member['email'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Email</div>
                                    <div class="col-md-8"><?php echo htmlspecialchars($member['email']); ?></div>
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($member['marital_status'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Marital Status</div>
                                    <div class="col-md-8"><?php echo htmlspecialchars($member['marital_status']); ?></div>
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($member['spouse_name'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Spouse Name</div>
                                    <div class="col-md-8"><?php echo htmlspecialchars($member['spouse_name']); ?></div>
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($member['emergency_contact_name']) || !empty($member['emergency_contact_number'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Emergency Contact</div>
                                    <div class="col-md-8">
                                        <?php if (!empty($member['emergency_contact_name'])): ?>
                                            <div><?php echo htmlspecialchars($member['emergency_contact_name']); ?></div>
                                        <?php endif; ?>
                                        <?php if (!empty($member['emergency_contact_number'])): ?>
                                            <div><?php echo htmlspecialchars($member['emergency_contact_number']); ?></div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($member['education_level'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Education</div>
                                    <div class="col-md-8"><?php echo htmlspecialchars($member['education_level']); ?></div>
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($member['church_role'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Church Role</div>
                                    <div class="col-md-8"><?php echo htmlspecialchars($member['church_role']); ?></div>
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($member['other_church_role'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Other Church Role</div>
                                    <div class="col-md-8"><?php echo htmlspecialchars($member['other_church_role']); ?></div>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Choir Information -->
                    <div class="col-md-6 mb-4">
                        <div class="card shadow info-card">
                            <div class="card-header py-3">
                                <h6 class="m-0 font-weight-bold text-primary">Choir Information</h6>
                            </div>
                            <div class="card-body">
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Member ID</div>
                                    <div class="col-md-8"><?php echo htmlspecialchars($member['id'] ?? 'Not Assigned'); ?></div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Voice Part</div>
                                    <div class="col-md-8"><?php echo htmlspecialchars($voicePartName); ?></div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Choir Role</div>
                                    <div class="col-md-8">
                                        <?php if (!empty($member['role'])): ?>
                                            <?php echo htmlspecialchars($member['role']); ?>
                                        <?php else: ?>
                                            N/A
                                        <?php endif; ?>
                                        <?php if (!empty($member['role_other'])): ?>
                                            <div class="text-muted small">Other: <?php echo htmlspecialchars($member['role_other']); ?></div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Leadership Position</div>
                                    <div class="col-md-8">
                                        <?php if (!empty($member['executive_position'])): ?>
                                            <?php echo htmlspecialchars($member['executive_position']); ?>
                                            <?php if (!empty($member['executive_position_other'])): ?>
                                                <div class="text-muted small">Other: <?php echo htmlspecialchars($member['executive_position_other']); ?></div>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            N/A
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Society</div>
                                    <div class="col-md-8"><?php echo htmlspecialchars($member['society_name'] ?? 'N/A'); ?></div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Circuit</div>
                                    <div class="col-md-8"><?php echo htmlspecialchars($member['circuit_name'] ?? 'N/A'); ?></div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Diocese</div>
                                    <div class="col-md-8"><?php echo htmlspecialchars($member['diocese_name'] ?? 'N/A'); ?></div>
                                </div>
                                <?php if (!empty($member['join_date'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Join Date</div>
                                    <div class="col-md-8"><?php echo date('F j, Y', strtotime($member['join_date'])); ?></div>
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($member['musical_skill_assessment'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Musical Skill</div>
                                    <div class="col-md-8"><?php echo nl2br(htmlspecialchars($member['musical_skill_assessment'])); ?></div>
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($member['musical_background'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Musical Background</div>
                                    <div class="col-md-8"><?php echo htmlspecialchars($member['musical_background']); ?></div>
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($member['instruments_played'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Instruments</div>
                                    <div class="col-md-8"><?php echo htmlspecialchars($member['instruments_played']); ?></div>
                                </div>
                                <?php endif; ?>
                                <?php if (isset($member['baptized']) || isset($member['confirmed']) || isset($member['communicant'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Church Status</div>
                                    <div class="col-md-8">
                                        <?php if (isset($member['baptized'])): ?>
                                            <div>Baptized: <?php echo $member['baptized'] === 'yes' ? 'Yes' : 'No'; ?></div>
                                        <?php endif; ?>
                                        <?php if (isset($member['confirmed'])): ?>
                                            <div>Confirmed: <?php echo $member['confirmed'] === 'yes' ? 'Yes' : 'No'; ?></div>
                                        <?php endif; ?>
                                        <?php if (isset($member['communicant'])): ?>
                                            <div>Communicant: <?php echo $member['communicant'] === 'yes' ? 'Yes' : 'No'; ?></div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($member['membership_type'])): ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Membership Type</div>
                                    <div class="col-md-8"><?php echo ucfirst(htmlspecialchars($member['membership_type'])); ?></div>
                                </div>
                                <?php endif; ?>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Model Choir</div>
                                    <div class="col-md-8">
                                        <span class="badge bg-<?php echo !empty($member['circuit_model_choir_member']) ? 'success' : 'secondary'; ?> me-1">
                                            Circuit Member: <?php echo !empty($member['circuit_model_choir_member']) ? 'Yes' : 'No'; ?>
                                        </span>
                                        <span class="badge bg-<?php echo !empty($member['diocesan_model_choir_member']) ? 'success' : 'secondary'; ?>">
                                            Diocese Member: <?php echo !empty($member['diocesan_model_choir_member']) ? 'Yes' : 'No'; ?>
                                        </span>
                                    </div>
                                </div>
                                <div class="row mb-2">
                                    <div class="col-md-4 info-label">Status</div>
                                    <div class="col-md-8">
                                        <span class="badge bg-<?php echo ($member['membership_status'] ?? 'inactive') === 'active' ? 'success' : (($member['membership_status'] ?? 'inactive') === 'inactive' ? 'warning' : 'danger'); ?>">
                                            <?php echo ucfirst(htmlspecialchars($member['membership_status'] ?? 'Inactive')); ?>
                                        </span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- System Account -->
                    <div class="col-md-6 mb-4">
                        <div class="card shadow info-card">
                            <div class="card-header py-3 bg-info text-white">
                                <h6 class="m-0 font-weight-bold">System Account</h6>
                            </div>
                            <div class="card-body">
                                <?php if (!empty($userAccount)): ?>
                                    <div class="row mb-2">
                                        <div class="col-md-4 info-label">Username</div>
                                        <div class="col-md-8"><?php echo htmlspecialchars($userAccount[0]['username']); ?></div>
                                    </div>
                                    <div class="row mb-2">
                                        <div class="col-md-4 info-label">Role</div>
                                        <div class="col-md-8">
                                            <span class="badge bg-primary"><?php echo htmlspecialchars($userAccount[0]['role_id']); ?></span>
                                        </div>
                                    </div>
                                    <div class="row mb-2">
                                        <div class="col-md-4 info-label">Status</div>
                                        <div class="col-md-8">
                                            <span class="badge bg-<?php echo $userAccount[0]['status'] === 'active' ? 'success' : ($userAccount[0]['status'] === 'inactive' ? 'warning' : 'danger'); ?>">
                                                <?php echo ucfirst(htmlspecialchars($userAccount[0]['status'])); ?>
                                            </span>
                                        </div>
                                    </div>
                                    <div class="row mb-2">
                                        <div class="col-md-4 info-label">Last Login</div>
                                        <div class="col-md-8">
                                            <?php echo !empty($userAccount[0]['last_login']) ? date('F j, Y g:i a', strtotime($userAccount[0]['last_login'])) : 'Never'; ?>
                                        </div>
                                    </div>
                                <?php else: ?>
                                    <div class="alert alert-info mb-0">
                                        <i class="bi bi-info-circle me-2"></i> This member does not have a system account.
                                        <?php if (hasRole('admin') || hasRole('manager')): ?>
                                            <div class="mt-3">
                                                <a href="../users/create.php?member_id=<?php echo $memberId; ?>" class="btn btn-sm btn-primary">
                                                    <i class="bi bi-person-plus"></i> Create Account
                                                </a>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Attendance Statistics -->
                    <div class="col-md-6 mb-4">
                        <div class="card shadow info-card">
                            <div class="card-header py-3 bg-success text-white">
                                <h6 class="m-0 font-weight-bold">Attendance Statistics</h6>
                            </div>
                            <div class="card-body">
                                <?php if (!empty($attendanceStats) && $attendanceStats[0]['total_events'] > 0): ?>
                                    <div class="row text-center mb-3">
                                        <div class="col-4">
                                            <div class="stat-card">
                                                <div class="stat-value text-success"><?php echo $attendanceStats[0]['present_count']; ?></div>
                                                <div class="stat-label">Present</div>
                                            </div>
                                        </div>
                                        <div class="col-4">
                                            <div class="stat-card">
                                                <div class="stat-value text-danger"><?php echo $attendanceStats[0]['absent_count']; ?></div>
                                                <div class="stat-label">Absent</div>
                                            </div>
                                        </div>
                                        <div class="col-4">
                                            <div class="stat-card">
                                                <div class="stat-value text-warning"><?php echo $attendanceStats[0]['excused_count']; ?></div>
                                                <div class="stat-label">Excused</div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="progress mb-2" style="height: 20px;">
                                        <div class="progress-bar bg-success" role="progressbar" 
                                             style="width: <?php echo $attendanceStats[0]['attendance_rate']; ?>%;" 
                                             aria-valuenow="<?php echo $attendanceStats[0]['attendance_rate']; ?>" 
                                             aria-valuemin="0" 
                                             aria-valuemax="100">
                                            <?php echo $attendanceStats[0]['attendance_rate']; ?>%
                                        </div>
                                    </div>
                                    <p class="text-center mb-4">
                                        <small>Overall attendance rate based on <?php echo $attendanceStats[0]['total_events']; ?> events</small>
                                    </p>
                                    
                                    <?php if (!empty($recentAttendance)): ?>
                                        <h6 class="mt-4">Recent Attendance</h6>
                                        <table class="table table-sm">
                                            <thead>
                                                <tr>
                                                    <th>Date</th>
                                                    <th>Status</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($recentAttendance as $record): ?>
                                                    <tr>
                                                        <td><?php echo date('M j, Y', strtotime($record['date'])); ?></td>
                                                        <td>
                                                            <span class="badge bg-<?php 
                                                                echo $record['status'] === 'present' ? 'success' : 
                                                                    ($record['status'] === 'absent' ? 'danger' : 'warning'); 
                                                            ?>">
                                                                <?php echo ucfirst($record['status']); ?>
                                                            </span>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    <?php endif; ?>
                                <?php else: ?>
                                    <div class="alert alert-info">
                                        No attendance records found for this member.
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Financial Information -->
                    <div class="col-md-6 mb-4">
                        <div class="card shadow info-card">
                            <div class="card-header py-3 bg-warning text-dark">
                                <h6 class="m-0 font-weight-bold">Financial Information</h6>
                            </div>
                            <div class="card-body">
                                <?php if (!empty($financialRecords)): ?>
                                    <div class="row text-center mb-4">
                                        <div class="col-12">
                                            <div class="stat-card">
                                                <div class="stat-value text-success">
                                                    GH₵ <?php echo number_format($totalContributions['total'] ?? 0, 2); ?>
                                                </div>
                                                <div class="stat-label">Total Contributions</div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <h6>Recent Transactions</h6>
                                    <table class="table table-sm">
                                        <thead>
                                            <tr>
                                                <th>Date</th>
                                                <th>Amount</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($financialRecords as $record): ?>
                                                <tr>
                                                    <td><?php echo date('M j, Y', strtotime($record['date'])); ?></td>
                                                    <td class="text-end">
                                                        GH₵ <?php echo number_format($record['amount'], 2); ?>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                <?php else: ?>
                                    <div class="alert alert-info">
                                        No financial records found for this member.
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <!-- Bootstrap JS Bundle with Popper -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <!-- jQuery -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
</body>
</html>
