<?php
/**
 * Job Portal - Job Details Page
 * GHAMECC Choir Management Platform
 * 
 * This page displays detailed information about a specific job and allows users to apply.
 */

// Debug: Log all requests to this page
error_log("=== JOB DETAILS PAGE REQUEST ===");
error_log("Request Method: " . $_SERVER['REQUEST_METHOD']);
error_log("GET params: " . print_r($_GET, true));
error_log("POST params: " . print_r($_POST, true));
error_log("Session data: " . print_r($_SESSION ?? [], true));

// Include database connection and functions
require_once '../config/database.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

// Initialize variables
$errors = [];
$success = false;
$applicationSubmitted = false;

// Check if job ID is provided
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header('Location: index.php');
    exit;
}

$jobId = (int)$_GET['id'];

// Get job details
$jobQuery = "SELECT j.*, c.name as category_name 
             FROM jobs j 
             LEFT JOIN job_categories c ON j.category_id = c.id 
             WHERE j.id = ?";
$job = fetchRow($jobQuery, [$jobId]);

// If job not found, redirect to index
if (!$job) {
    header('Location: index.php');
    exit;
}

// Get job media
$mediaQuery = "SELECT * FROM job_media WHERE job_id = ? ORDER BY id ASC";
$jobMedia = fetchRows($mediaQuery, [$jobId]);

// Check if user is logged in
$isLoggedIn = isset($_SESSION['user_id']) && isset($_SESSION['user_email']);
$memberId = $_SESSION['member_id'] ?? null;

// Check if user has already applied for this job (only if logged in)
$applicationSubmitted = false;
if ($isLoggedIn) {
    $applicationQuery = "SELECT id FROM job_applications WHERE job_id = ? AND job_portal_user_id = ?";
    $existingApplication = fetchRow($applicationQuery, [$jobId, $_SESSION['user_id']]);
    $applicationSubmitted = !empty($existingApplication);
}

// Handle job application submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['apply'])) {
    logInfo("Job application submission started", [
        'job_id' => $jobId,
        'user_id' => $_SESSION['user_id'] ?? 'not_set',
        'member_id' => $_SESSION['member_id'] ?? 'not_set',
        'post_data_keys' => array_keys($_POST),
        'files_data_keys' => array_keys($_FILES)
    ]);
    
    // Check if user is logged in
    if (!$isLoggedIn) {
        logError("Application submission failed - user not logged in", [
            'job_id' => $jobId,
            'session_data' => $_SESSION
        ]);
        $errors[] = 'You must be logged in to apply for jobs';
    } else {
        logInfo("User login verified", [
            'user_id' => $_SESSION['user_id'],
            'member_id' => $memberId,
            'has_member_id' => !empty($memberId)
        ]);
        
        // Validate form data
        $coverLetter = isset($_POST['cover_letter']) ? trim($_POST['cover_letter']) : '';
        
        logInfo("Form data validation", [
            'cover_letter_length' => strlen($coverLetter),
            'cover_letter_empty' => empty($coverLetter)
        ]);
        
        if (empty($coverLetter)) {
            logError("Validation failed - cover letter empty", [
                'job_id' => $jobId,
                'user_id' => $_SESSION['user_id']
            ]);
            $errors[] = 'Cover letter is required';
        }
        
        // Check if user has already applied for this job
        $checkStmt = $conn->prepare("SELECT id FROM job_applications WHERE job_id = ? AND job_portal_user_id = ?");
        $checkParams = [$jobId, $_SESSION['user_id']];
        
        if (!$checkStmt) {
            logError("Failed to prepare check statement", [
                'error' => $conn->error,
                'job_id' => $jobId,
                'user_id' => $_SESSION['user_id'],
                'member_id' => $memberId
            ]);
            $errors[] = 'Database error occurred';
        } else {
            $checkStmt->bind_param("ii", ...$checkParams);
            $checkResult = $checkStmt->execute();
            
            if (!$checkResult) {
                logError("Failed to execute check query", [
                    'error' => $checkStmt->error,
                    'job_id' => $jobId,
                    'user_id' => $_SESSION['user_id'],
                    'member_id' => $memberId
                ]);
                $errors[] = 'Database error occurred';
            } else {
                $existingApplication = $checkStmt->get_result();
                if ($existingApplication->num_rows > 0) {
                    logInfo("User has already applied for this job", [
                        'job_id' => $jobId,
                        'user_id' => $_SESSION['user_id'],
                        'member_id' => $memberId
                    ]);
                    $errors[] = 'You have already applied for this job';
                } else {
                    logInfo("No existing application found - proceeding", [
                        'job_id' => $jobId,
                        'user_id' => $_SESSION['user_id'],
                        'member_id' => $memberId
                    ]);
                }
            }
            $checkStmt->close();
        }
        
        // Insert application into database if no errors
        if (empty($errors)) {
            logInfo("Starting database insertion", [
                'job_id' => $jobId,
                'user_id' => $_SESSION['user_id'],
                'member_id' => $memberId,
                'cover_letter_length' => strlen($coverLetter)
            ]);
            
            $stmt = $conn->prepare("INSERT INTO job_applications (job_id, job_portal_user_id, cover_letter, status, created_at) VALUES (?, ?, ?, 'pending', NOW())");
            $insertParams = [$jobId, $_SESSION['user_id'], $coverLetter];
            
            if (!$stmt) {
                logError("Failed to prepare insert statement", [
                    'error' => $conn->error,
                    'job_id' => $jobId,
                    'user_id' => $_SESSION['user_id'],
                    'member_id' => $memberId
                ]);
                $errors[] = 'Database error occurred while preparing application';
            } else {
                logInfo("Insert statement prepared successfully", [
                    'job_id' => $jobId,
                    'user_id' => $_SESSION['user_id'],
                    'member_id' => $memberId,
                    'using_member_id' => !empty($memberId)
                ]);
                
                $stmt->bind_param("iis", ...$insertParams);
                $result = $stmt->execute();
                
                if (!$result) {
                    logError("Failed to insert job application", [
                        'error' => $stmt->error,
                        'errno' => $stmt->errno,
                        'job_id' => $jobId,
                        'user_id' => $_SESSION['user_id'],
                        'member_id' => $memberId,
                        'cover_letter_length' => strlen($coverLetter)
                    ]);
                    $errors[] = 'Failed to submit application: ' . $stmt->error;
                } else {
                    $applicationId = $conn->insert_id;
                    logInfo("Job application inserted successfully", [
                        'application_id' => $applicationId,
                        'job_id' => $jobId,
                        'user_id' => $_SESSION['user_id'],
                        'member_id' => $memberId
                    ]);
                    
                    // Handle file uploads
                    if (isset($_FILES['application_media']) && !empty($_FILES['application_media']['name'][0])) {
                        logInfo("Processing file uploads", [
                            'application_id' => $applicationId,
                            'file_count' => count($_FILES['application_media']['name'])
                        ]);
                        
                        $uploadDir = '../uploads/jobs/' . $_SESSION['user_id'] . '/';
                        if (!is_dir($uploadDir)) {
                            if (!mkdir($uploadDir, 0755, true)) {
                                logError("Failed to create upload directory", [
                                    'directory' => $uploadDir,
                                    'application_id' => $applicationId
                                ]);
                                $errors[] = 'Failed to create upload directory';
                            } else {
                                logInfo("Upload directory created", ['directory' => $uploadDir]);
                            }
                        }
                        
                        if (empty($errors)) {
                            foreach ($_FILES['application_media']['name'] as $key => $fileName) {
                                if (!empty($fileName)) {
                                    $fileSize = $_FILES['application_media']['size'][$key];
                                    $fileTmpName = $_FILES['application_media']['tmp_name'][$key];
                                    $fileError = $_FILES['application_media']['error'][$key];
                                    
                                    logInfo("Processing file upload", [
                                        'file_name' => $fileName,
                                        'file_size' => $fileSize,
                                        'file_error' => $fileError,
                                        'key' => $key
                                    ]);
                                    
                                    if ($fileError === UPLOAD_ERR_OK) {
                                        $fileExtension = pathinfo($fileName, PATHINFO_EXTENSION);
                                        $uniqueFileName = time() . '_' . $key . '.' . $fileExtension;
                                        $filePath = $uploadDir . $uniqueFileName;
                                        
                                        if (move_uploaded_file($fileTmpName, $filePath)) {
                                            logInfo("File uploaded successfully", [
                                                'original_name' => $fileName,
                                                'saved_path' => $filePath,
                                                'application_id' => $applicationId
                                            ]);
                                            
                                            // Insert media record
                                            $mediaStmt = $conn->prepare("INSERT INTO application_media (application_id, media_type, file_path, file_name, file_size, created_at) VALUES (?, 'document', ?, ?, ?, NOW())");
                                            if ($mediaStmt) {
                                                $mediaStmt->bind_param("issi", $applicationId, $filePath, $fileName, $fileSize);
                                                $mediaResult = $mediaStmt->execute();
                                                
                                                if ($mediaResult) {
                                                    logInfo("Media record inserted", [
                                                        'media_id' => $conn->insert_id,
                                                        'application_id' => $applicationId,
                                                        'file_name' => $fileName
                                                    ]);
                                                } else {
                                                    logError("Failed to insert media record", [
                                                        'error' => $mediaStmt->error,
                                                        'application_id' => $applicationId,
                                                        'file_name' => $fileName
                                                    ]);
                                                }
                                                $mediaStmt->close();
                                            } else {
                                                logError("Failed to prepare media insert statement", [
                                                    'error' => $conn->error,
                                                    'application_id' => $applicationId
                                                ]);
                                            }
                                        } else {
                                            logError("Failed to move uploaded file", [
                                                'from' => $fileTmpName,
                                                'to' => $filePath,
                                                'application_id' => $applicationId
                                            ]);
                                            $errors[] = 'Failed to upload file: ' . $fileName;
                                        }
                                    } else {
                                        logError("File upload error", [
                                            'file_name' => $fileName,
                                            'error_code' => $fileError,
                                            'application_id' => $applicationId
                                        ]);
                                        $errors[] = 'Error uploading file: ' . $fileName;
                                    }
                                }
                            }
                        }
                    } else {
                        logInfo("No files to upload", ['application_id' => $applicationId]);
                    }
                    
                    if (empty($errors)) {
                        logInfo("Job application submission completed successfully", [
                            'application_id' => $applicationId,
                            'job_id' => $jobId,
                            'user_id' => $_SESSION['user_id'],
                            'member_id' => $memberId
                        ]);
                        $success = 'Your application has been submitted successfully!';
                    }
                }
                $stmt->close();
            }
        } else {
            logError("Application submission failed due to validation errors", [
                'errors' => $errors,
                'job_id' => $jobId,
                'user_id' => $_SESSION['user_id'] ?? 'not_set'
            ]);
        }
    }
    
    logInfo("Job application submission process completed", [
        'job_id' => $jobId,
        'success' => isset($success),
        'error_count' => count($errors),
        'errors' => $errors
    ]);
}

// Get similar jobs
$similarJobsQuery = "SELECT j.*, c.name as category_name 
                    FROM jobs j 
                    LEFT JOIN job_categories c ON j.category_id = c.id 
                    WHERE j.category_id = ? AND j.id != ? AND j.status = 'open' 
                    ORDER BY j.created_at DESC 
                    LIMIT 3";
$similarJobs = fetchRows($similarJobsQuery, [$job['category_id'], $jobId]);

// Page title
$pageTitle = htmlspecialchars($job['title']) . " - GHAMECC Job Portal";
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $pageTitle; ?></title>
    
    <?php include '../includes/favicon.php'; ?>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <!-- Lightbox CSS for image gallery -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/lightbox2/2.11.3/css/lightbox.min.css" rel="stylesheet">
    <!-- Summernote WYSIWYG Editor -->
    <link href="https://cdn.jsdelivr.net/npm/summernote@0.8.18/dist/summernote-bs4.min.css" rel="stylesheet">
    <style>
        .job-header {
            background-color: #f8f9fa;
            border-radius: 10px;
            padding: 30px;
            margin-bottom: 30px;
            box-shadow: 0 0 15px rgba(0,0,0,0.05);
        }
        .job-meta {
            display: inline-block;
            margin-right: 20px;
            margin-bottom: 10px;
            color: #6c757d;
        }
        .job-meta i {
            margin-right: 5px;
            color: #0d6efd;
        }
        .job-section {
            margin-bottom: 30px;
        }
        .media-gallery {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin-top: 15px;
        }
        .media-item {
            width: 150px;
            height: 150px;
            object-fit: cover;
            border-radius: 5px;
            transition: transform 0.3s;
        }
        .media-item:hover {
            transform: scale(1.05);
        }
        .similar-job {
            transition: transform 0.3s;
        }
        .similar-job:hover {
            transform: translateY(-5px);
        }
        .application-form {
            background-color: #f8f9fa;
            border-radius: 10px;
            padding: 30px;
            margin-top: 30px;
            box-shadow: 0 0 15px rgba(0,0,0,0.05);
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <?php include 'includes/job-portal-navbar.php'; ?>
    
    <!-- Main Content -->
    <div class="container py-5">
        <!-- Success Message -->
        <?php if ($success): ?>
        <div class="alert alert-success mb-4">
            <i class="fas fa-check-circle"></i> Your application has been submitted successfully! The employer will contact you if they're interested.
        </div>
        <?php endif; ?>
        
        <!-- Error Messages -->
        <?php if (!empty($errors)): ?>
        <div class="alert alert-danger mb-4">
            <ul class="mb-0">
                <?php foreach ($errors as $error): ?>
                <li><?php echo $error; ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
        <?php endif; ?>
        
        <div class="row">
            <!-- Job Details Column -->
            <div class="col-lg-8">
                <!-- Job Header -->
                <div class="job-header">
                    <div class="d-flex justify-content-between align-items-start">
                        <div>
                            <h1 class="mb-3"><?php echo htmlspecialchars($job['title']); ?></h1>
                            <h5 class="text-muted mb-3"><?php echo htmlspecialchars($job['company_name']); ?></h5>
                        </div>
                        <span class="badge bg-<?php echo ($job['status'] === 'open') ? 'success' : 'danger'; ?> fs-6">
                            <?php echo ucfirst($job['status']); ?>
                        </span>
                    </div>
                    
                    <div class="mb-3">
                        <span class="job-meta"><i class="fas fa-map-marker-alt"></i> <?php echo htmlspecialchars($job['location']); ?></span>
                        <span class="job-meta"><i class="fas fa-tag"></i> <?php echo htmlspecialchars($job['category_name']); ?></span>
                        <?php if (!empty($job['salary_range'])): ?>
                        <span class="job-meta"><i class="fas fa-money-bill-wave"></i> <?php echo htmlspecialchars($job['salary_range']); ?></span>
                        <?php endif; ?>
                        <span class="job-meta"><i class="fas fa-calendar-alt"></i> Deadline: <?php echo date('F j, Y', strtotime($job['deadline'])); ?></span>
                        <span class="job-meta"><i class="far fa-clock"></i> Posted <?php echo timeElapsed($job['created_at']); ?></span>
                    </div>
                    
                    <?php if ($job['status'] === 'open'): ?>
                    <div class="mt-4">
                        <?php if ($isLoggedIn && !$applicationSubmitted): ?>
                        <a href="#apply-section" class="btn btn-primary">Apply Now</a>
                        <?php elseif ($applicationSubmitted): ?>
                        <button class="btn btn-success" disabled>Application Submitted</button>
                        <?php else: ?>
                        <a href="login.php?redirect=job-details.php?id=<?php echo $jobId; ?>" class="btn btn-primary">Login to Apply</a>
                        <?php endif; ?>
                        <a href="index.php" class="btn btn-outline-secondary ms-2">Back to Jobs</a>
                    </div>
                    <?php else: ?>
                    <div class="alert alert-warning mt-3">
                        <i class="fas fa-exclamation-triangle"></i> This job is no longer accepting applications.
                    </div>
                    <?php endif; ?>
                </div>
                
                <!-- Job Description -->
                <div class="job-section">
                    <h3>Job Description</h3>
                    <hr>
                    <div class="job-description">
                        <?php echo cleanText($job['description'], true); ?>
                    </div>
                </div>
                
                <!-- Job Requirements -->
                <?php if (!empty($job['requirements'])): ?>
                <div class="job-section">
                    <h3>Requirements</h3>
                    <hr>
                    <div class="job-requirements">
                        <?php echo cleanText($job['requirements'], true); ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Job Media -->
                <?php if (!empty($jobMedia)): ?>
                <div class="job-section">
                    <h3>Media Gallery</h3>
                    <hr>
                    <div class="media-gallery">
                        <?php foreach ($jobMedia as $media): ?>
                            <?php if ($media['media_type'] === 'image'): ?>
                                <a href="../<?php echo $media['file_path']; ?>" data-lightbox="job-gallery" data-title="<?php echo htmlspecialchars($media['file_name']); ?>">
                                    <img src="../<?php echo $media['file_path']; ?>" alt="<?php echo htmlspecialchars($media['file_name']); ?>" class="media-item">
                                </a>
                            <?php elseif ($media['media_type'] === 'video'): ?>
                                <div class="card" style="width: 150px;">
                                    <div class="card-body text-center">
                                        <i class="fas fa-film fa-2x text-primary"></i>
                                        <p class="card-text mt-2"><?php echo htmlspecialchars(truncateText($media['file_name'], 15)); ?></p>
                                        <a href="../<?php echo $media['file_path']; ?>" class="btn btn-sm btn-primary" target="_blank">View</a>
                                    </div>
                                </div>
                            <?php else: ?>
                                <div class="card" style="width: 150px;">
                                    <div class="card-body text-center">
                                        <i class="fas fa-file-alt fa-2x text-primary"></i>
                                        <p class="card-text mt-2"><?php echo htmlspecialchars(truncateText($media['file_name'], 15)); ?></p>
                                        <a href="../<?php echo $media['file_path']; ?>" class="btn btn-sm btn-primary" target="_blank">Download</a>
                                    </div>
                                </div>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Application Form -->
                <?php if ($job['status'] === 'open' && $isLoggedIn && !$applicationSubmitted): ?>
                <div id="apply-section" class="application-form">
                    <h3>Apply for this Job</h3>
                    <hr>
                    <form action="job-details.php?id=<?php echo $jobId; ?>#apply-section" method="POST" enctype="multipart/form-data">
                        <div class="mb-3">
                            <label for="cover_letter" class="form-label">Cover Letter</label>
                            <textarea class="form-control summernote" id="cover_letter" name="cover_letter" rows="6" required></textarea>
                            <div class="form-text">Explain why you're a good fit for this position.</div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="application_media" class="form-label">Upload Resume, Portfolio, or Other Documents</label>
                            <input type="file" class="form-control" id="application_media" name="application_media[]" multiple>
                            <div class="form-text">You can upload multiple files. Maximum size per file: 5MB.</div>
                        </div>
                        
                        <div class="d-grid gap-2">
                            <button type="submit" name="apply" class="btn btn-primary btn-lg">Submit Application</button>
                        </div>
                    </form>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- Sidebar Column -->
            <div class="col-lg-4">
                <!-- Contact Information -->
                <div class="card mb-4">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0">Contact Information</h5>
                    </div>
                    <div class="card-body">
                        <p><i class="fas fa-envelope me-2 text-primary"></i> <?php echo htmlspecialchars($job['contact_email']); ?></p>
                        <p><i class="fas fa-phone me-2 text-primary"></i> <?php echo htmlspecialchars($job['contact_phone']); ?></p>
                    </div>
                </div>
                
                <!-- Similar Jobs -->
                <?php if (!empty($similarJobs)): ?>
                <div class="card mb-4">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0">Similar Jobs</h5>
                    </div>
                    <div class="card-body p-0">
                        <ul class="list-group list-group-flush">
                            <?php foreach ($similarJobs as $similarJob): ?>
                            <li class="list-group-item similar-job">
                                <h6 class="mb-1">
                                    <a href="job-details.php?id=<?php echo $similarJob['id']; ?>" class="text-decoration-none">
                                        <?php echo htmlspecialchars($similarJob['title']); ?>
                                    </a>
                                </h6>
                                <p class="mb-1 text-muted small">
                                    <?php echo htmlspecialchars($similarJob['company_name']); ?> • 
                                    <?php echo htmlspecialchars($similarJob['location']); ?>
                                </p>
                                <div class="d-flex justify-content-between align-items-center">
                                    <small class="text-muted">
                                        Posted <?php echo timeElapsed($similarJob['created_at']); ?>
                                    </small>
                                    <span class="badge bg-primary"><?php echo htmlspecialchars($similarJob['category_name']); ?></span>
                                </div>
                            </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    <div class="card-footer text-center">
                        <a href="index.php?category=<?php echo $job['category_id']; ?>" class="btn btn-outline-primary btn-sm">
                            View All in <?php echo htmlspecialchars($job['category_name']); ?>
                        </a>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Post a Job Card -->
                <div class="card">
                    <div class="card-header bg-secondary text-white">
                        <h5 class="mb-0">Have a Job to Offer?</h5>
                    </div>
                    <div class="card-body">
                        <p>Post your job opportunity and reach qualified GHAMECC members with the skills you need.</p>
                        <div class="d-grid">
                            <a href="post-job.php" class="btn btn-secondary">Post a Job</a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Footer -->
    <footer class="bg-dark text-white py-4 mt-5">
        <div class="container">
            <div class="row">
                <div class="col-md-4 mb-3">
                    <h5>GHAMECC Job Portal</h5>
                    <p>A platform for Ghana Methodist Church Choir members to find and post job opportunities.</p>
                </div>
                <div class="col-md-4 mb-3">
                    <h5>Quick Links</h5>
                    <ul class="list-unstyled">
                        <li><a href="index.php" class="text-white">Home</a></li>
                        <li><a href="post-job.php" class="text-white">Post a Job</a></li>
                        <li><a href="talent-showcase.php" class="text-white">Talent Showcase</a></li>
                        <li><a href="about.php" class="text-white">About</a></li>
                    </ul>
                </div>
                <div class="col-md-4 mb-3">
                    <h5>Contact</h5>
                    <address>
                        <p><i class="fas fa-envelope me-2"></i> info@ghamecc.org</p>
                        <p><i class="fas fa-phone me-2"></i> +233 XX XXX XXXX</p>
                    </address>
                </div>
            </div>
            <hr>
            <div class="text-center">
                <p>&copy; <?php echo date('Y'); ?> GHAMECC Job Portal. All rights reserved.</p>
            </div>
        </div>
    </footer>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <!-- jQuery -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <!-- Lightbox JS -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/lightbox2/2.11.3/js/lightbox.min.js"></script>
    <!-- Summernote JS -->
    <script src="https://cdn.jsdelivr.net/npm/summernote@0.8.18/dist/summernote-bs4.min.js"></script>
    <script>
        $(document).ready(function() {
            $('.summernote').summernote({
                placeholder: 'Start typing here...',
                height: 200,
                toolbar: [
                    ['style', ['bold', 'italic', 'underline', 'clear']],
                    ['font', ['strikethrough']],
                    ['para', ['ul', 'ol']],
                    ['insert', ['link']]
                ]
            });
            
            // Initialize lightbox
            lightbox.option({
                'resizeDuration': 200,
                'wrapAround': true
            });
        });
    </script>
</body>
</html>
