<?php
/**
 * Create Test Jobs for Job Approval Testing
 * Temporary script to create sample pending jobs for testing
 */

// Include database connection and functions
require_once 'config/database.php';
require_once 'includes/functions.php';
require_once 'includes/auth.php';

// Check if user is logged in and is admin
if (!isLoggedIn() || !hasRole(['admin'])) {
    echo "Access denied. Admin access required.";
    exit;
}

echo "<h2>Creating Test Jobs for Job Approval</h2>";

// First, check if job_categories exist
$categories = fetchRows("SELECT * FROM job_categories LIMIT 5");
if (empty($categories)) {
    echo "No job categories found. Creating default categories...<br>";
    
    $defaultCategories = [
        ['Information Technology', 'Software development, IT support, network administration, etc.'],
        ['Music & Arts', 'Music production, performance, teaching, etc.'],
        ['Administration', 'Office administration, secretarial work, etc.'],
        ['Education', 'Teaching, tutoring, training, etc.'],
        ['Other', 'Other job categories not listed']
    ];
    
    foreach ($defaultCategories as $cat) {
        $insertCat = "INSERT INTO job_categories (name, description) VALUES (?, ?)";
        executeQuery($insertCat, $cat);
    }
    
    $categories = fetchRows("SELECT * FROM job_categories LIMIT 5");
    echo "Created " . count($categories) . " job categories.<br>";
}

// Get some societies for testing
$societies = fetchRows("SELECT * FROM societies LIMIT 3");
if (empty($societies)) {
    echo "No societies found in database.<br>";
} else {
    echo "Found " . count($societies) . " societies for testing.<br>";
}

// Get some members for testing
$members = fetchRows("SELECT * FROM members LIMIT 3");
if (empty($members)) {
    echo "No members found in database.<br>";
} else {
    echo "Found " . count($members) . " members for testing.<br>";
}

// Create test jobs
$testJobs = [
    [
        'title' => 'Web Developer Position',
        'description' => 'We are looking for a skilled web developer to join our team. The candidate should have experience in PHP, MySQL, HTML, CSS, and JavaScript.',
        'requirements' => 'Bachelor\'s degree in Computer Science or related field. 2+ years of web development experience.',
        'location' => 'Accra, Ghana',
        'salary_range' => 'GHS 3,000 - 5,000',
        'company_name' => 'Tech Solutions Ghana',
        'contact_email' => 'hr@techsolutions.com.gh',
        'contact_phone' => '+233 24 123 4567',
        'deadline' => date('Y-m-d', strtotime('+30 days')),
        'category_id' => $categories[0]['id'],
        'society_id' => !empty($societies) ? $societies[0]['id'] : null,
        'member_id' => !empty($members) ? $members[0]['id'] : null,
        'approval_status' => 'pending'
    ],
    [
        'title' => 'Music Teacher - Piano',
        'description' => 'Seeking an experienced piano teacher to provide lessons to children and adults. Must be patient and have excellent communication skills.',
        'requirements' => 'Music degree or equivalent experience. Minimum 3 years teaching experience.',
        'location' => 'Kumasi, Ghana',
        'salary_range' => 'GHS 2,000 - 3,500',
        'company_name' => 'Harmony Music School',
        'contact_email' => 'info@harmonymusic.com.gh',
        'contact_phone' => '+233 20 987 6543',
        'deadline' => date('Y-m-d', strtotime('+21 days')),
        'category_id' => $categories[1]['id'],
        'society_id' => !empty($societies) ? $societies[1]['id'] ?? $societies[0]['id'] : null,
        'member_id' => !empty($members) ? $members[1]['id'] ?? $members[0]['id'] : null,
        'approval_status' => 'pending'
    ],
    [
        'title' => 'Administrative Assistant',
        'description' => 'Looking for an organized administrative assistant to support daily office operations. Duties include filing, data entry, and customer service.',
        'requirements' => 'High school diploma required. Proficiency in Microsoft Office. Strong organizational skills.',
        'location' => 'Takoradi, Ghana',
        'salary_range' => 'GHS 1,500 - 2,500',
        'company_name' => 'Western Region Office',
        'contact_email' => 'admin@westernoffice.com.gh',
        'contact_phone' => '+233 31 456 7890',
        'deadline' => date('Y-m-d', strtotime('+14 days')),
        'category_id' => $categories[2]['id'],
        'society_id' => !empty($societies) ? $societies[2]['id'] ?? $societies[0]['id'] : null,
        'member_id' => !empty($members) ? $members[2]['id'] ?? $members[0]['id'] : null,
        'approval_status' => 'pending'
    ]
];

echo "<h3>Creating Test Jobs:</h3>";

$insertJobQuery = "INSERT INTO jobs (title, description, requirements, location, salary_range, company_name, 
                   contact_email, contact_phone, deadline, category_id, society_id, member_id, approval_status, created_at) 
                   VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";

$createdCount = 0;
foreach ($testJobs as $job) {
    try {
        $result = executeQuery($insertJobQuery, [
            $job['title'],
            $job['description'],
            $job['requirements'],
            $job['location'],
            $job['salary_range'],
            $job['company_name'],
            $job['contact_email'],
            $job['contact_phone'],
            $job['deadline'],
            $job['category_id'],
            $job['society_id'],
            $job['member_id'],
            $job['approval_status']
        ]);
        
        if ($result) {
            echo "✓ Created job: " . htmlspecialchars($job['title']) . "<br>";
            $createdCount++;
        } else {
            echo "✗ Failed to create job: " . htmlspecialchars($job['title']) . "<br>";
        }
    } catch (Exception $e) {
        echo "✗ Error creating job '" . htmlspecialchars($job['title']) . "': " . $e->getMessage() . "<br>";
    }
}

echo "<br><strong>Summary:</strong> Created $createdCount test jobs.<br>";

// Verify the jobs were created
echo "<h3>Verification - Pending Jobs Count:</h3>";
try {
    $pendingCount = fetchRow("SELECT COUNT(*) as count FROM jobs WHERE approval_status = 'pending'");
    echo "Total pending jobs in database: " . $pendingCount['count'] . "<br>";
    
    if ($pendingCount['count'] > 0) {
        echo "<br><a href='job-approval.php' target='_blank'>→ Go to Job Approval Page</a><br>";
        echo "<a href='debug_job_approval.php' target='_blank'>→ Run Debug Script</a><br>";
    }
} catch (Exception $e) {
    echo "Error verifying jobs: " . $e->getMessage() . "<br>";
}

echo "<br><em>Note: This is a temporary script for testing. You can delete it after testing is complete.</em>";
?>
