<?php
/**
 * Admin Settings Management
 * GHAMECC Choir Management Platform
 */

// Include authentication functions
require_once '../includes/auth.php';
require_once '../config/database.php';

// Require login and admin role
requireLogin();
requireRole('admin');

// Get user information
$userId = $_SESSION['user_id'];
$userFullName = $_SESSION['full_name'] ?? $_SESSION['email'] ?? 'User';

// Set base path for includes
$basePath = '..';

// Process form submissions
$message = '';
$messageType = '';

// Define default settings if not already in database
$defaultSettings = [
    'site_name' => 'GHAMECC Choir Management',
    'site_description' => 'Ghana Methodist Church Choir Management Platform',
    'contact_email' => 'admin@ghamecc.org',
    'contact_phone' => '+233 000 000 000',
    'items_per_page' => '20',
    'enable_notifications' => '1',
    'enable_sms' => '0',
    'enable_email' => '1',
    'maintenance_mode' => '0',
    'maintenance_message' => 'System is currently under maintenance. Please check back later.',
    'footer_text' => ' ' . date('Y') . ' GHAMECC Choir Management Platform. All rights reserved.'
];

// Handle settings update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_settings') {
    // Update each setting
    foreach ($_POST as $key => $value) {
        if ($key !== 'action' && array_key_exists($key, $defaultSettings)) {
            // Check if setting already exists
            $stmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
            $stmt->bind_param("s", $key);
            $stmt->execute();
            $result = $stmt->get_result();
            $stmt->close();
            
            if ($result->num_rows > 0) {
                // Update existing setting
                $stmt = $conn->prepare("UPDATE settings SET setting_value = ?, updated_by = ?, updated_at = NOW() WHERE setting_key = ?");
                $stmt->bind_param("sis", $value, $userId, $key);
            } else {
                // Insert new setting
                $stmt = $conn->prepare("INSERT INTO settings (setting_key, setting_value, created_by) VALUES (?, ?, ?)");
                $stmt->bind_param("ssi", $key, $value, $userId);
            }
            
            $stmt->execute();
            $stmt->close();
        }
    }
    
    $message = 'Settings updated successfully.';
    $messageType = 'success';
}

// Get current settings
$settings = $defaultSettings;

// Check if settings table exists
$tableExists = false;
$result = executeQuery($conn, "SHOW TABLES LIKE 'settings'");
if ($result && $result->num_rows > 0) {
    $tableExists = true;
}

if ($tableExists) {
    $result = executeQuery($conn, "SELECT setting_key, setting_value FROM settings");
    
    if ($result) {
        $rows = fetchRows($result);
        foreach ($rows as $row) {
            $settings[$row['setting_key']] = $row['setting_value'];
        }
        $result->free();
    }
} else {
    // Create settings table if it doesn't exist
    executeQuery($conn, "CREATE TABLE IF NOT EXISTS settings (
        id INT AUTO_INCREMENT PRIMARY KEY,
        setting_key VARCHAR(100) NOT NULL UNIQUE,
        setting_value TEXT NOT NULL,
        created_by INT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_by INT,
        updated_at TIMESTAMP NULL,
        FOREIGN KEY (created_by) REFERENCES users(id),
        FOREIGN KEY (updated_by) REFERENCES users(id)
    )");
    
    // Insert default settings
    foreach ($defaultSettings as $key => $value) {
        $stmt = $conn->prepare("INSERT INTO settings (setting_key, setting_value, created_by) VALUES (?, ?, ?)");
        $stmt->bind_param("ssi", $key, $value, $userId);
        $stmt->execute();
        $stmt->close();
    }
    
    $message = 'Settings table created with default values.';
    $messageType = 'info';
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>System Settings - GHAMECC Choir Management</title>
    
    <?php include '../includes/favicon.php'; ?>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <!-- Custom CSS -->
    <link href="../assets/css/style.css" rel="stylesheet">
</head>
<body>
    <!-- Navigation -->
    <?php include '../includes/navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <?php include '../includes/sidebar.php'; ?>
            
            <!-- Main Content -->
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">System Settings</h1>
                </div>
                
                <?php if (!empty($message)): ?>
                <div class="alert alert-<?php echo $messageType; ?> alert-dismissible fade show" role="alert">
                    <?php echo $message; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php endif; ?>
                
                <!-- Settings Form -->
                <div class="card shadow mb-4">
                    <div class="card-header py-3">
                        <h6 class="m-0 font-weight-bold text-primary">System Configuration</h6>
                    </div>
                    <div class="card-body">
                        <form method="post" action="">
                            <input type="hidden" name="action" value="update_settings">
                            
                            <h5 class="mb-3">General Settings</h5>
                            <div class="row mb-4">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="site_name" class="form-label">Site Name</label>
                                        <input type="text" class="form-control" id="site_name" name="site_name" 
                                               value="<?php echo htmlspecialchars($settings['site_name']); ?>" required>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="site_description" class="form-label">Site Description</label>
                                        <input type="text" class="form-control" id="site_description" name="site_description" 
                                               value="<?php echo htmlspecialchars($settings['site_description']); ?>">
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row mb-4">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="contact_email" class="form-label">Contact Email</label>
                                        <input type="email" class="form-control" id="contact_email" name="contact_email" 
                                               value="<?php echo htmlspecialchars($settings['contact_email']); ?>">
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="contact_phone" class="form-label">Contact Phone</label>
                                        <input type="text" class="form-control" id="contact_phone" name="contact_phone" 
                                               value="<?php echo htmlspecialchars($settings['contact_phone']); ?>">
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row mb-4">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="items_per_page" class="form-label">Items Per Page</label>
                                        <input type="number" class="form-control" id="items_per_page" name="items_per_page" 
                                               value="<?php echo htmlspecialchars($settings['items_per_page']); ?>" min="5" max="100">
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="footer_text" class="form-label">Footer Text</label>
                                        <input type="text" class="form-control" id="footer_text" name="footer_text" 
                                               value="<?php echo htmlspecialchars($settings['footer_text']); ?>">
                                    </div>
                                </div>
                            </div>
                            
                            <h5 class="mb-3">Notification Settings</h5>
                            <div class="row mb-4">
                                <div class="col-md-4">
                                    <div class="form-check form-switch mb-3">
                                        <input class="form-check-input" type="checkbox" id="enable_notifications" name="enable_notifications" value="1"
                                               <?php echo $settings['enable_notifications'] ? 'checked' : ''; ?>>
                                        <label class="form-check-label" for="enable_notifications">Enable Notifications</label>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-check form-switch mb-3">
                                        <input class="form-check-input" type="checkbox" id="enable_sms" name="enable_sms" value="1"
                                               <?php echo $settings['enable_sms'] ? 'checked' : ''; ?>>
                                        <label class="form-check-label" for="enable_sms">Enable SMS Notifications</label>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-check form-switch mb-3">
                                        <input class="form-check-input" type="checkbox" id="enable_email" name="enable_email" value="1"
                                               <?php echo $settings['enable_email'] ? 'checked' : ''; ?>>
                                        <label class="form-check-label" for="enable_email">Enable Email Notifications</label>
                                    </div>
                                </div>
                            </div>
                            
                            <h5 class="mb-3">Maintenance Settings</h5>
                            <div class="row mb-4">
                                <div class="col-md-6">
                                    <div class="form-check form-switch mb-3">
                                        <input class="form-check-input" type="checkbox" id="maintenance_mode" name="maintenance_mode" value="1"
                                               <?php echo $settings['maintenance_mode'] ? 'checked' : ''; ?>>
                                        <label class="form-check-label" for="maintenance_mode">Enable Maintenance Mode</label>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="maintenance_message" class="form-label">Maintenance Message</label>
                                        <textarea class="form-control" id="maintenance_message" name="maintenance_message" rows="3"><?php echo htmlspecialchars($settings['maintenance_message']); ?></textarea>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save me-2"></i>Save Settings
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <!-- Bootstrap JS Bundle with Popper -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <!-- jQuery -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
</body>
</html>
